#!/usr/bin/env bash
set -euo pipefail
ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
META="$ROOT/.codex/meta"
AUTO_COMMIT="$META/git-autosave.enabled.flag"
AUTO_PUSH="$META/git-autopush.enabled.flag"
BRANCH_SESSION="$META/git-branch-per-session.enabled.flag"
COOLDOWN_FILE="$META/git-checkpoint.cooldown-minutes.flag"
LAST_FILE="$META/git-checkpoint.last.flag"
SESSION_BRANCH_FILE="$META/git-session-branch.flag"
HIST="$ROOT/.codex/state/history.md"

force=0
if [[ "${1:-}" == "--force" ]]; then
  force=1
  shift
fi
MSG="${1:-checkpoint}"

cd "$ROOT"

enabled() {
  local f="$1"
  [[ -f "$f" ]] && grep -qx "ENABLED" "$f"
}

read_int() {
  local f="$1" default="$2"
  if [[ -f "$f" ]]; then
    local v
    v="$(tr -d ' \t\r\n' < "$f" || true)"
    [[ "$v" =~ ^[0-9]+$ ]] && { echo "$v"; return 0; }
  fi
  echo "$default"
}

now_epoch() {
  python3 - <<'PY'
import time
print(int(time.time()))
PY
}

if ! git rev-parse --is-inside-work-tree >/dev/null 2>&1; then
  echo "SKIP: not a git repository."
  exit 0
fi

if ! enabled "$AUTO_COMMIT"; then
  echo "SKIP: autosave disabled (create .codex/meta/git-autosave.enabled.flag with ENABLED)."
  exit 0
fi

mkdir -p "$META" "$(dirname "$HIST")"

# Cooldown check
cooldown_min="$(read_int "$COOLDOWN_FILE" 10)"
cooldown_sec="$((cooldown_min * 60))"
if [[ "$force" -eq 0 && "$cooldown_sec" -gt 0 && -f "$LAST_FILE" ]]; then
  last="$(tr -d ' \t\r\n' < "$LAST_FILE" || true)"
  if [[ "$last" =~ ^[0-9]+$ ]]; then
    now="$(now_epoch)"
    elapsed="$((now - last))"
    if [[ "$elapsed" -lt "$cooldown_sec" ]]; then
      echo "SKIP: cooldown active (${cooldown_min}m). Use --force to bypass."
      exit 0
    fi
  fi
fi

# Branch-per-session: if enabled, and on main/master, create/checkout session branch once
if enabled "$BRANCH_SESSION"; then
  current_branch="$(git rev-parse --abbrev-ref HEAD)"
  if [[ "$current_branch" == "main" || "$current_branch" == "master" ]]; then
    if [[ ! -f "$SESSION_BRANCH_FILE" ]]; then
      ts="$(date +%Y%m%d-%H%M)"
      shortid="$(python3 - <<'PY'
import secrets
print(secrets.token_hex(3))
PY
)"
      session_branch="codex/${ts}-${shortid}"
      echo "$session_branch" > "$SESSION_BRANCH_FILE"
      echo "Creating session branch: $session_branch"
      git checkout -b "$session_branch"
    else
      session_branch="$(tr -d ' \t\r\n' < "$SESSION_BRANCH_FILE" || true)"
      if [[ -n "$session_branch" && "$current_branch" != "$session_branch" ]]; then
        # Don't hard-switch if there are uncommitted changes
        if git diff --quiet && git diff --cached --quiet; then
          echo "Switching to recorded session branch: $session_branch"
          git checkout "$session_branch" || true
        else
          echo "WARN: session branch recorded ($session_branch) but working tree is dirty; staying on $current_branch."
        fi
      fi
    fi
  fi
fi

# No changes?
if git diff --quiet && git diff --cached --quiet; then
  echo "SKIP: no changes to commit."
  exit 0
fi

git add -A

if git diff --cached --quiet; then
  echo "SKIP: nothing staged after add."
  exit 0
fi

git commit -m "$MSG" || { echo "FAIL: commit failed."; exit 1; }

# Update cooldown timestamp
now="$(now_epoch)"
echo "$now" > "$LAST_FILE"

# Log history
if ! grep -q "^## Git Checkpoints" "$HIST" 2>/dev/null; then
  cat >> "$HIST" <<'EOF'

## Git Checkpoints
- (none)

EOF
fi

python3 - "$HIST" <<'PY'
import pathlib, re, sys
p = pathlib.Path(sys.argv[1])
t = p.read_text(encoding="utf-8") if p.exists() else ""
t2 = re.sub(r"(## Git Checkpoints\s*\n)- \(none\)\s*\n", r"\1", t, count=1)
p.write_text(t2, encoding="utf-8")
PY

branch="$(git rev-parse --abbrev-ref HEAD)"
sha="$(git rev-parse --short HEAD)"
echo "- $(date -Is) — $MSG — $branch@$sha" >> "$HIST"

# Optional push
if enabled "$AUTO_PUSH"; then
  if git remote get-url origin >/dev/null 2>&1; then
    branch="$(git rev-parse --abbrev-ref HEAD)"
    echo "Pushing to origin/$branch..."
    if git push -u origin "$branch"; then
      echo "OK: pushed."
    else
      echo "WARN: push failed (commit is local)."
      exit 0
    fi
  else
    echo "WARN: autopush enabled but no origin remote configured."
  fi
else
  echo "OK: commit created (autopush disabled)."
fi

exit 0
