#!/usr/bin/env bash
set -euo pipefail
ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
AUTO_COMMIT="$ROOT/.codex/meta/git-autosave.enabled.flag"
AUTO_PUSH="$ROOT/.codex/meta/git-autopush.enabled.flag"
HIST="$ROOT/.codex/state/history.md"
MSG="${1:-checkpoint}"

cd "$ROOT"

enabled() {
  local f="$1"
  [[ -f "$f" ]] && grep -qx "ENABLED" "$f"
}

if ! git rev-parse --is-inside-work-tree >/dev/null 2>&1; then
  echo "SKIP: not a git repository."
  exit 0
fi

if ! enabled "$AUTO_COMMIT"; then
  echo "SKIP: autosave disabled (create .codex/meta/git-autosave.enabled.flag with ENABLED)."
  exit 0
fi

# no changes?
if git diff --quiet && git diff --cached --quiet; then
  echo "SKIP: no changes to commit."
  exit 0
fi

git add -A

# If still nothing staged, skip
if git diff --cached --quiet; then
  echo "SKIP: nothing staged after add."
  exit 0
fi

# commit
git commit -m "$MSG" || { echo "FAIL: commit failed."; exit 1; }

# log history
mkdir -p "$(dirname "$HIST")"
if ! grep -q "^## Git Checkpoints" "$HIST" 2>/dev/null; then
  cat >> "$HIST" <<'EOF'

## Git Checkpoints
- (none)

EOF
fi

python3 - "$HIST" <<'PY'
import pathlib, re, sys
p = pathlib.Path(sys.argv[1])
t = p.read_text(encoding="utf-8") if p.exists() else ""
t2 = re.sub(r"(## Git Checkpoints\s*\n)- \(none\)\s*\n", r"\1", t, count=1)
p.write_text(t2, encoding="utf-8")
PY

echo "- $(date -Is) — $MSG" >> "$HIST"

# optional push
if enabled "$AUTO_PUSH"; then
  if git remote get-url origin >/dev/null 2>&1; then
    # Determine current branch
    branch="$(git rev-parse --abbrev-ref HEAD)"
    echo "Pushing to origin/$branch..."
    if git push origin "$branch"; then
      echo "OK: pushed."
    else
      echo "WARN: push failed (commit is local)."
      exit 0
    fi
  else
    echo "WARN: autopush enabled but no origin remote configured."
  fi
else
  echo "OK: commit created (autopush disabled)."
fi

exit 0
