#!/usr/bin/env bash
set -euo pipefail

ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
CODEX="$ROOT/.codex"
FIX="${1:-}"

if [[ ! -d "$CODEX" ]]; then
  echo "OK: .codex does not exist."
  exit 0
fi

# Allowed extensions outside artifacts
allowed_re='(\.md$|\.sh$|\.flag$|/\.keep$)'
violations=()

while IFS= read -r -d '' f; do
  # skip directories
  [[ -f "$f" ]] || continue
  # ignore artifacts
  if [[ "$f" == "$CODEX/artifacts/"* ]]; then
    continue
  fi
  if [[ ! "$f" =~ $allowed_re ]]; then
    violations+=("$f")
  fi
done < <(find "$CODEX" -type f -print0)

if [[ ${#violations[@]} -eq 0 ]]; then
  echo "OK: .codex lint passed (no disallowed files outside artifacts)."
  exit 0
fi

echo "FAIL: .codex contains disallowed files (outside .codex/artifacts):"
for v in "${violations[@]}"; do
  echo " - ${v#$ROOT/}"
done

echo ""
echo "Policy: .codex/meta/codex-lint-policy.md"
echo "Fix guidance:"
echo " - move files into .codex/artifacts/misc/ OR out of .codex entirely"
echo " - remove from git if tracked: git rm --cached <path>"

if [[ "$FIX" == "--fix" ]]; then
  mkdir -p "$CODEX/artifacts/misc"
  for v in "${violations[@]}"; do
    bn="$(basename "$v")"
    dest="$CODEX/artifacts/misc/$bn"
    echo "Moving: ${v#$ROOT/} -> .codex/artifacts/misc/$bn"
    mv -f "$v" "$dest"
  done
  echo "Fix complete. Re-run lint."
  exit 0
fi

exit 1
