#!/usr/bin/env bash
set -euo pipefail
ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
MASTER="$ROOT/MASTER_SPEC.md"
REPORT="$ROOT/.codex/state/spec-run-report.md"

if [[ ! -f "$MASTER" ]]; then
  echo "ERROR: MASTER_SPEC.md not found at repo root."
  exit 1
fi

# Lightweight validation: ensure expected headings exist (best-effort, non-blocking)
missing=0
for h in "## 1) Project name" "## 2) One-liner + Project summary" "## 3) Problem / why" "## 4) Solution" "## 5) Users + core workflows" "## 6) Pages and copy" "## 7) Product requirements" "## 8) Acceptance criteria" "## 9) Unknowns allowed"; do
  if ! grep -qF "$h" "$MASTER"; then
    echo "WARN: Missing heading: $h"
    missing=1
  fi
done

# Determine project slug
PROJECT_SLUG="${PROJECT_SLUG:-$(basename "$ROOT" | tr '[:upper:]' '[:lower:]' | tr -cd 'a-z0-9-_')}"
SPEC_DIR="$ROOT/.specify/specs/$PROJECT_SLUG"
MEM_DIR="$ROOT/.specify/memory"

# Ensure .specify exists (prefer specify CLI when available)
if [[ ! -d "$ROOT/.specify" ]]; then
  if command -v specify >/dev/null 2>&1; then
    echo "Initializing Spec Kit via specify CLI..."
    set +e
    specify init --here --ai codex
    rc=$?
    set -e
    if [[ $rc -ne 0 ]]; then
      echo "WARN: specify init failed (rc=$rc). Falling back to minimal .specify scaffolding."
      mkdir -p "$MEM_DIR" "$SPEC_DIR"
    fi
  else
    echo "specify CLI not found; creating minimal .specify scaffolding."
    mkdir -p "$MEM_DIR" "$SPEC_DIR"
  fi
fi

mkdir -p "$MEM_DIR" "$SPEC_DIR"
mkdir -p "$ROOT/.codex/state" "$ROOT/.codex/memory" "$ROOT/.codex/meta"

# Create constitution if missing
CONSTITUTION="$MEM_DIR/constitution.md"
if [[ ! -f "$CONSTITUTION" ]]; then
  cat > "$CONSTITUTION" <<'EOF'
# Spec Kit Constitution (generated by MemoryKit Spec YOLO)

Core rules:
- Read MASTER_SPEC.md as the single source input.
- Generate spec/plan/tasks without asking the user questions.
- Any missing info must be explicitly labeled as [ASSUMPTION] with rationale, or left as TBD.
- Do not silently invent facts.
- Keep an audit trail in .codex/memory/decisions.md and .codex/state/history.md.

END
EOF
fi

SPEC="$SPEC_DIR/spec.md"
PLAN="$SPEC_DIR/plan.md"
TASKS="$SPEC_DIR/tasks.md"

cat > "$SPEC" <<EOF
# Spec — $PROJECT_SLUG
*Generated from:* \`MASTER_SPEC.md\`  
*Generated at:* $(date -Is)

---

## Master Input (verbatim)
> NOTE: This section is copied from MASTER_SPEC.md. Do not edit it here; edit MASTER_SPEC.md instead.

\`\`\`md
$(cat "$MASTER")
\`\`\`

---

## Derived Specification
Derive concrete implementation details from the master input.

### Goals
- TBD

### Non-goals
- TBD

### Users / Actors
- TBD

### Core workflows
- TBD

### Pages / IA (information architecture)
- TBD

### Requirements
- TBD

### Integrations
- TBD

### Tech preferences
- TBD (or "choose best default")

### Non-functional requirements
- Performance: TBD
- Security: TBD
- Accessibility: TBD
- Observability: TBD

### Acceptance criteria
- TBD

---

## Assumptions (quarantined)
Add only if necessary; each must be tagged:

- [ASSUMPTION] ...
  - Rationale: ...

---

## Unknowns / TBD
List anything that cannot be safely assumed.

END
EOF

cat > "$PLAN" <<EOF
# Plan — $PROJECT_SLUG
*Generated at:* $(date -Is)

This plan is derived from \`MASTER_SPEC.md\` + \`spec.md\`.

## Milestones
1) Prototype / design-first (if applicable)
2) Implementation
3) Testing + QA
4) Docs + release

## Execution notes
- Use design-first for UI when applicable.
- Use autonomous loops only with hard checks.

## Assumptions used
- (keep in sync with spec.md assumptions)

END
EOF

cat > "$TASKS" <<EOF
# Tasks — $PROJECT_SLUG
*Generated at:* $(date -Is)

Generate a task list from the spec. Prefer small, verifiable tasks.

## Task list
- [ ] Create/refine design-first prototype OR confirm memory-only path
- [ ] Confirm page hierarchy / IA aligns with master input
- [ ] Implement core features
- [ ] Add tests (smoke + e2e where useful)
- [ ] Produce screenshot artifact (for UI work)
- [ ] Update docs (README + run/test steps)

## Verification checks
- Tests pass (where applicable)
- App builds/runs (where applicable)
- Screenshot exists at .codex/artifacts/screenshots/ (UI projects)

END
EOF

cat > "$REPORT" <<EOF
# Spec Run Report

- Date: $(date -Is)
- Project slug: $PROJECT_SLUG
- Input: MASTER_SPEC.md
- Headings complete: $([[ "$missing" == "0" ]] && echo "yes" || echo "no (see warnings above)")
- Outputs:
  - $SPEC
  - $PLAN
  - $TASKS
  - $CONSTITUTION

Rules:
- No user Q&A during Spec YOLO.
- Unknowns must be `TBD` or tagged `[ASSUMPTION]` with rationale.

END
EOF

echo "Wrote:"
echo " - $SPEC"
echo " - $PLAN"
echo " - $TASKS"
echo " - $REPORT"
