#!/usr/bin/env bash
set -euo pipefail
ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
MASTER="$ROOT/MASTER_SPEC.md"
REPORT="$ROOT/.codex/state/spec-run-report.md"

if [[ ! -f "$MASTER" ]]; then
  echo "ERROR: MASTER_SPEC.md not found at repo root."
  exit 1
fi

# Determine project slug
PROJECT_SLUG="${PROJECT_SLUG:-$(basename "$ROOT" | tr '[:upper:]' '[:lower:]' | tr -cd 'a-z0-9-_')}"
SPEC_DIR="$ROOT/.specify/specs/$PROJECT_SLUG"
MEM_DIR="$ROOT/.specify/memory"

# Ensure .specify exists (prefer specify CLI when available)
if [[ ! -d "$ROOT/.specify" ]]; then
  if command -v specify >/dev/null 2>&1; then
    echo "Initializing Spec Kit via specify CLI..."
    set +e
    specify init --here --ai codex
    rc=$?
    set -e
    if [[ $rc -ne 0 ]]; then
      echo "WARN: specify init failed (rc=$rc). Falling back to minimal .specify scaffolding."
      mkdir -p "$MEM_DIR" "$SPEC_DIR"
    fi
  else
    echo "specify CLI not found; creating minimal .specify scaffolding."
    mkdir -p "$MEM_DIR" "$SPEC_DIR"
  fi
fi

mkdir -p "$MEM_DIR" "$SPEC_DIR"
mkdir -p "$ROOT/.codex/state" "$ROOT/.codex/memory" "$ROOT/.codex/meta"

# Create constitution if missing
CONSTITUTION="$MEM_DIR/constitution.md"
if [[ ! -f "$CONSTITUTION" ]]; then
  cat > "$CONSTITUTION" <<'EOF'
# Spec Kit Constitution (generated by MemoryKit Spec YOLO)

Core rules:
- Read MASTER_SPEC.md as the single source input.
- Generate spec/plan/tasks without asking the user questions.
- Any missing info must be explicitly labeled as [ASSUMPTION] with rationale, or left as TBD.
- Do not silently invent facts.
- Keep an audit trail in .codex/memory/decisions.md and .codex/state/history.md.

END
EOF
fi

SPEC="$SPEC_DIR/spec.md"
PLAN="$SPEC_DIR/plan.md"
TASKS="$SPEC_DIR/tasks.md"

# Write spec/plan/tasks templates (idempotent overwrite is OK for Spec YOLO runs)
cat > "$SPEC" <<EOF
# Spec — $PROJECT_SLUG
*Generated from:* \`MASTER_SPEC.md\`  
*Generated at:* $(date -Is)

---

## Master Input (verbatim)
> NOTE: This section is copied from MASTER_SPEC.md. Do not edit it here; edit MASTER_SPEC.md instead.

\`\`\`md
$(cat "$MASTER")
\`\`\`

---

## Derived Specification
(Write the concrete spec here, derived from the master input.)

### Goals
- TBD

### Non-goals
- TBD

### Users / Actors
- TBD

### Core workflows
- TBD

### Requirements
- TBD

### Non-functional requirements
- Performance: TBD
- Security: TBD
- Accessibility: TBD
- Observability: TBD

### Acceptance criteria
- TBD

---

## Assumptions (quarantined)
Add only if necessary; each must be tagged:

- [ASSUMPTION] ...
  - Rationale: ...

---

## Open questions
If something cannot be safely assumed, list it here as TBD.

END
EOF

cat > "$PLAN" <<EOF
# Plan — $PROJECT_SLUG
*Generated at:* $(date -Is)

This plan is derived from \`MASTER_SPEC.md\` + \`spec.md\`.

## Milestones
1) Prototype / design-first
2) Implementation
3) Testing + QA
4) Docs + release

## Execution notes
- Use design-first for UI when applicable.
- Use autonomous loops only with hard checks.

## Assumptions used
- (keep in sync with spec.md assumptions)

END
EOF

cat > "$TASKS" <<EOF
# Tasks — $PROJECT_SLUG
*Generated at:* $(date -Is)

Generate a task list from the spec. Prefer small, verifiable tasks.

## Task list
- [ ] Create or refine design-first prototype
- [ ] Approve UI (decision log + ui-approved flag)
- [ ] Implement core features
- [ ] Add tests (smoke + e2e where useful)
- [ ] Produce screenshot artifact
- [ ] Update docs (README + run/test steps)

## Verification checks
- Tests pass
- App builds/runs
- Screenshot exists at .codex/artifacts/screenshots/

END
EOF

# Write report
cat > "$REPORT" <<EOF
# Spec Run Report

- Date: $(date -Is)
- Project slug: $PROJECT_SLUG
- Input: MASTER_SPEC.md
- Outputs:
  - $SPEC
  - $PLAN
  - $TASKS
  - $CONSTITUTION

Notes:
- This run does not ask the user questions.
- Any missing info must be recorded as [ASSUMPTION] or left TBD.

END
EOF

echo "Wrote:"
echo " - $SPEC"
echo " - $PLAN"
echo " - $TASKS"
echo " - $REPORT"
