#!/usr/bin/env bash
set -euo pipefail
ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
MASTER="$ROOT/MASTER_SPEC.md"
REPORT="$ROOT/.codex/state/spec-run-report.md"

if [[ ! -f "$MASTER" ]]; then
  echo "ERROR: MASTER_SPEC.md not found at repo root."
  exit 1
fi

extract_name() {
  awk '
    BEGIN{found=0}
    /^## 1\) Project name[[:space:]]*$/ {found=1; next}
    found==1 {
      if ($0 ~ /^##[[:space:]]/) {exit}
      if ($0 ~ /^[[:space:]]*$/) {next}
      gsub(/^[[:space:]]+|[[:space:]]+$/, "", $0)
      print $0
      exit
    }
  ' "$MASTER"
}

slugify() {
  echo "$1" | tr '[:upper:]' '[:lower:]' | sed -E 's/[ _]+/-/g; s/[^a-z0-9-]+//g; s/-+/-/g; s/^-+//; s/-+$//'
}

NAME_FROM_MASTER="$(extract_name || true)"
DERIVED_SLUG=""
if [[ -n "${PROJECT_SLUG:-}" ]]; then
  DERIVED_SLUG="$PROJECT_SLUG"
elif [[ -n "$NAME_FROM_MASTER" ]]; then
  DERIVED_SLUG="$(slugify "$NAME_FROM_MASTER")"
fi

if [[ -z "$DERIVED_SLUG" ]]; then
  DERIVED_SLUG="$(basename "$ROOT" | tr '[:upper:]' '[:lower:]' | tr -cd 'a-z0-9-_')"
fi

PROJECT_SLUG="$DERIVED_SLUG"
SPEC_DIR="$ROOT/.specify/specs/$PROJECT_SLUG"
MEM_DIR="$ROOT/.specify/memory"

if [[ ! -d "$ROOT/.specify" ]]; then
  if command -v specify >/dev/null 2>&1; then
    specify init --here --ai codex || true
  fi
fi

mkdir -p "$MEM_DIR" "$SPEC_DIR" "$ROOT/.codex/state"

CONSTITUTION="$MEM_DIR/constitution.md"
if [[ ! -f "$CONSTITUTION" ]]; then
  cat > "$CONSTITUTION" <<'EOF'
# Spec Kit Constitution (generated by MemoryKit Spec YOLO)

- Read MASTER_SPEC.md as the single source input.
- Generate spec/plan/tasks without asking the user questions.
- Unknowns must be [ASSUMPTION] with rationale, or TBD.
- Keep an audit trail.

END
EOF
fi

SPEC="$SPEC_DIR/spec.md"
PLAN="$SPEC_DIR/plan.md"
TASKS="$SPEC_DIR/tasks.md"

cat > "$SPEC" <<EOF
# Spec — $PROJECT_SLUG
*Generated from:* \`MASTER_SPEC.md\`  
*Generated at:* $(date -Is)

\`\`\`md
$(cat "$MASTER")
\`\`\`

## Derived Specification
- TBD

## Assumptions (quarantined)
- [ASSUMPTION] ...

END
EOF

cat > "$PLAN" <<EOF
# Plan — $PROJECT_SLUG
Generated at: $(date -Is)

- TBD

END
EOF

cat > "$TASKS" <<EOF
# Tasks — $PROJECT_SLUG
Generated at: $(date -Is)

- [ ] TBD

END
EOF

cat > "$REPORT" <<EOF
# Spec Run Report

- Date: $(date -Is)
- Extracted project name: ${NAME_FROM_MASTER:-"(none)"}
- Project slug: $PROJECT_SLUG

Outputs:
- $SPEC
- $PLAN
- $TASKS

END
EOF

echo "Wrote spec outputs in .specify/ for $PROJECT_SLUG"
