#!/usr/bin/env bash
set -euo pipefail
ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
LEVEL="${1:-}"
TARGET="${2:-}"
NOTE="${3:-}"
LOCKS_DIR="$ROOT/.codex/meta/locks"
REGISTRY="$ROOT/.codex/meta/component-locks.md"
HIST="$ROOT/.codex/state/history.md"

if [[ -z "$LEVEL" || -z "$TARGET" ]]; then
  echo "Usage: lock-component.sh <SOFT|HARD> <PATH_OR_GLOB> \"note\""
  exit 1
fi

if [[ "$LEVEL" != "SOFT" && "$LEVEL" != "HARD" ]]; then
  echo "ERROR: LEVEL must be SOFT or HARD"
  exit 1
fi

mkdir -p "$LOCKS_DIR" "$(dirname "$HIST")" "$(dirname "$REGISTRY")"

ts="$(date +%Y%m%d-%H%M%S)"
rid="$(python3 - <<'PY'
import secrets
print(secrets.token_hex(3))
PY
)"
lock_id="lock-${ts}-${rid}"

lock_file="$LOCKS_DIR/${lock_id}.lock"
{
  echo "ID=$lock_id"
  echo "LEVEL=$LEVEL"
  echo "TARGET=$TARGET"
  echo "NOTE=${NOTE:-}"
  echo "CREATED_AT=$(date -Is)"
} > "$lock_file"

if [[ ! -f "$REGISTRY" ]]; then
  cat > "$REGISTRY" <<'EOF'
# Component Locks (prevent “helpful” rewrites)

## Current locks
- (none)

END
EOF
fi

python3 - "$REGISTRY" <<'PY'
import pathlib, re, sys
p = pathlib.Path(sys.argv[1])
t = p.read_text(encoding="utf-8")
t2 = re.sub(r"(## Current locks\s*\n)- \(none\)\s*\n", r"\1", t, count=1)
p.write_text(t2, encoding="utf-8")
PY

echo "- $(date -Is) — $lock_id — $LEVEL — $TARGET — ${NOTE:-}" >> "$REGISTRY"

if ! grep -q "^## Lock Events" "$HIST" 2>/dev/null; then
  cat >> "$HIST" <<'EOF'

## Lock Events
- (none)

EOF
fi
python3 - "$HIST" <<'PY'
import pathlib, re, sys
p = pathlib.Path(sys.argv[1])
t = p.read_text(encoding="utf-8") if p.exists() else ""
t2 = re.sub(r"(## Lock Events\s*\n)- \(none\)\s*\n", r"\1", t, count=1)
p.write_text(t2, encoding="utf-8")
PY

echo "- $(date -Is) — LOCKED — $lock_id — $LEVEL — $TARGET — ${NOTE:-}" >> "$HIST"

echo "Locked: $lock_id ($LEVEL) -> $TARGET"
