#!/usr/bin/env bash
set -euo pipefail

ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
REPORT="$ROOT/.codex/state/doctor-report.md"

MODEFILE="$ROOT/.codex/meta/project-mode.md"
INITFLAG="$ROOT/.codex/meta/initialized.flag"
UIFLAG="$ROOT/.codex/meta/ui-approved.flag"
LOCKS="$ROOT/.codex/meta/memory-locks.md"
CAP="$ROOT/.codex/memory/capabilities.md"

SPECDIR="$ROOT/.specify"
CONSTITUTION="$ROOT/.specify/memory/constitution.md"
SPECSOURCES="$ROOT/.codex/meta/spec-sources.md"

mode="unset"
if [[ -f "$MODEFILE" ]]; then
  mode="$(grep -E '^Current:' "$MODEFILE" | sed 's/^Current:[[:space:]]*//' | tr -d '\r' || true)"
  [[ -z "$mode" ]] && mode="unset"
fi

ui="n/a"
if [[ "$mode" == "design-first" || "$mode" == "yolo" ]]; then
  [[ -f "$UIFLAG" ]] && ui="approved" || ui="not-approved"
fi

lock_count=0
lock_lines=""
if [[ -f "$LOCKS" ]]; then
  lock_lines="$(grep -E '^[[:space:]]*-[[:space:]]*(FILE|SECTION):' "$LOCKS" || true)"
  [[ -n "$lock_lines" ]] && lock_count="$(echo "$lock_lines" | wc -l | tr -d ' ')"
fi

spec_present="no"
spec_health="n/a"
active_spec="n/a"
if [[ -d "$SPECDIR" ]]; then
  spec_present="yes"
  [[ -f "$CONSTITUTION" ]] && spec_health="ok" || spec_health="missing-constitution"
  if [[ -f "$SPECSOURCES" ]]; then
    active_spec="$(grep -E '^[[:space:]]*active_feature_spec:' -A1 "$SPECSOURCES" | tail -n1 | sed 's/^[[:space:]]*-[[:space:]]*//' | tr -d '\r')"
    [[ -z "$active_spec" ]] && active_spec="(unset)"
  fi
fi

cap_ok="missing"
if [[ -f "$CAP" ]]; then
  grep -q "Generated" "$CAP" && cap_ok="ok" || cap_ok="suspicious"
fi

{
  echo "# Memory Doctor Report"
  echo "Generated by \`memory-doctor\`. Overwritten each run."
  echo ""
  echo "## Summary"
  echo "- Date: $(date -Is)"
  echo "- Initialized: $([[ -f "$INITFLAG" ]] && echo yes || echo no)"
  echo "- Mode: $mode"
  echo "- UI gate: $ui"
  echo "- Locks: $lock_count"
  echo "- Capabilities: $cap_ok"
  echo "- Spec Kit present: $spec_present"
  echo "- Spec Kit health: $spec_health"
  echo "- Active feature spec: $active_spec"
  echo ""
  echo "## Active locks"
  if [[ $lock_count -eq 0 ]]; then
    echo "- (none)"
  else
    echo "$lock_lines" | sed 's/^[[:space:]]*-/ -/'
  fi
} > "$REPORT"

echo "Wrote $REPORT"
