#!/usr/bin/env bash
set -euo pipefail

ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
REPORT="$ROOT/.codex/state/doctor-report.md"
MODEFILE="$ROOT/.codex/meta/project-mode.md"
INITFLAG="$ROOT/.codex/meta/initialized.flag"
UIFLAG="$ROOT/.codex/meta/ui-approved.flag"
LOCKS="$ROOT/.codex/meta/memory-locks.md"
CAP="$ROOT/.codex/memory/capabilities.md"

allowed_modes=("design-first" "yolo" "existing" "memory-only")

req_files=(
  "$ROOT/.codex/agents.md"
  "$ROOT/.codex/brief.md"
  "$ROOT/.codex/meta/mutation-guards.md"
  "$ROOT/.codex/meta/memory-locks.md"
  "$ROOT/.codex/meta/skill-trigger-spec.md"
  "$ROOT/.codex/meta/design-workflow.md"
  "$ROOT/.codex/meta/yolo-runbook.md"
  "$ROOT/.codex/meta/tools.md"
  "$ROOT/.codex/memory/decisions.md"
  "$ROOT/.codex/state/history.md"
  "$ROOT/.codex/memory/capabilities.md"
)

missing=0
for f in "${req_files[@]}"; do
  [[ -f "$f" ]] || missing=$((missing+1))
done

mode="unset"
if [[ -f "$MODEFILE" ]]; then
  mode="$(grep -E '^Current:' "$MODEFILE" | sed 's/^Current:[[:space:]]*-[[:space:]]*//' | tr -d '\r' || true)"
  [[ -z "$mode" ]] && mode="unset"
fi

mode_ok="unknown"
if [[ "$mode" == "unset" ]]; then
  mode_ok="unset"
else
  mode_ok="invalid"
  for m in "${allowed_modes[@]}"; do
    [[ "$mode" == "$m" ]] && mode_ok="ok"
  done
fi

cap_ok="unknown"
if [[ -f "$CAP" ]]; then
  grep -q "Generated" "$CAP" && cap_ok="ok" || cap_ok="suspicious"
fi

ui="n/a"
if [[ "$mode" == "design-first" || "$mode" == "yolo" ]]; then
  [[ -f "$UIFLAG" ]] && ui="approved" || ui="not-approved"
fi

lock_count=0
lock_lines=""
if [[ -f "$LOCKS" ]]; then
  lock_lines="$(grep -E '^[[:space:]]*-[[:space:]]*(FILE|SECTION):' "$LOCKS" || true)"
  [[ -n "$lock_lines" ]] && lock_count="$(echo "$lock_lines" | wc -l | tr -d ' ')"
fi

{
  echo "# Memory Doctor Report"
  echo "Generated by \`memory-doctor\`. Overwritten each run."
  echo ""
  echo "## Summary"
  echo "- Date: $(date -Is)"
  echo "- Missing required files: $missing"
  echo "- Mode: $mode ($mode_ok)"
  echo "- Initialized flag present: $([[ -f "$INITFLAG" ]] && echo yes || echo no)"
  echo "- Capabilities file: $cap_ok"
  echo "- UI gate (design-first/yolo only): $ui"
  echo "- Locks: $lock_count"
  echo ""
  echo "## Active locks"
  if [[ $lock_count -eq 0 ]]; then
    echo "- (none)"
  else
    echo "$lock_lines" | sed 's/^[[:space:]]*-/ -/'
  fi
} > "$REPORT"

echo "Wrote $REPORT"
