#!/usr/bin/env bash
set -euo pipefail

ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")/../.." && pwd)"
REPORT="$ROOT/.codex/state/doctor-report.md"
MODEFILE="$ROOT/.codex/meta/project-mode.md"
INITFLAG="$ROOT/.codex/meta/initialized.flag"
UIFLAG="$ROOT/.codex/meta/ui-approved.flag"

req_files=(
  "$ROOT/.codex/agents.md"
  "$ROOT/.codex/brief.md"
  "$ROOT/.codex/meta/mutation-guards.md"
  "$ROOT/.codex/meta/skill-trigger-spec.md"
  "$ROOT/.codex/meta/design-workflow.md"
  "$ROOT/.codex/memory/decisions.md"
  "$ROOT/.codex/state/history.md"
  "$ROOT/.codex/memory/capabilities.md"
)

missing=0
for f in "${req_files[@]}"; do
  if [[ ! -f "$f" ]]; then
    missing=$((missing+1))
  fi
done

mode="unset"
if [[ -f "$MODEFILE" ]]; then
  mode="$(grep -E '^Current:' "$MODEFILE" | sed 's/^Current:[[:space:]]*-[[:space:]]*//' | tr -d '\r' || true)"
  [[ -z "$mode" ]] && mode="unset"
fi

cap_ok="unknown"
if [[ -f "$ROOT/.codex/memory/capabilities.md" ]]; then
  if grep -q "Generated" "$ROOT/.codex/memory/capabilities.md"; then
    cap_ok="ok"
  else
    cap_ok="suspicious"
  fi
fi

ui="n/a"
if [[ "$mode" == "design-first" ]]; then
  if [[ -f "$UIFLAG" ]]; then
    ui="approved"
  else
    ui="not-approved"
  fi
fi

{
  echo "# Memory Doctor Report"
  echo "Generated by \`memory-doctor\`. Overwritten each run."
  echo ""
  echo "## Summary"
  echo "- Date: $(date -Is)"
  echo "- Missing required files: $missing"
  echo "- Mode: $mode"
  echo "- Initialized flag present: $([[ -f "$INITFLAG" ]] && echo yes || echo no)"
  echo "- Capabilities file: $cap_ok"
  echo "- UI gate (design-first only): $ui"
  echo ""
  echo "## Missing files"
  if [[ $missing -eq 0 ]]; then
    echo "- (none)"
  else
    for f in "${req_files[@]}"; do
      [[ -f "$f" ]] || echo "- $f"
    done
  fi
} > "$REPORT"

echo "Wrote $REPORT"
